//--------------------------------------------------------------------------
//
//  Software for MSP430 based e-meters.
//
//  THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR
//  REPRESENTATIONS, EITHER EXPRESS, IMPLIED OR STATUTORY,
//  INCLUDING ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
//  FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY OR
//  COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE.
//  TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET
//  POSSESSION, AND NON-INFRINGEMENT OF ANY THIRD PARTY
//  INTELLECTUAL PROPERTY RIGHTS WITH REGARD TO THE PROGRAM OR
//  YOUR USE OF THE PROGRAM.
//
//  IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
//  CONSEQUENTIAL OR INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY
//  THEORY OF LIABILITY AND WHETHER OR NOT TI HAS BEEN ADVISED
//  OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY OUT
//  OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM.
//  EXCLUDED DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF
//  REMOVAL OR REINSTALLATION, COMPUTER TIME, LABOR COSTS, LOSS
//  OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS, OR LOSS OF
//  USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S
//  AGGREGATE LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF
//  YOUR USE OF THE PROGRAM EXCEED FIVE HUNDRED DOLLARS
//  (U.S.$500).
//
//  Unless otherwise stated, the Program written and copyrighted
//  by Texas Instruments is distributed as "freeware".  You may,
//  only under TI's copyright in the Program, use and modify the
//  Program without any charge or restriction.  You may
//  distribute to third parties, provided that you transfer a
//  copy of this license to the third party and the third party
//  agrees to these terms by its first use of the Program. You
//  must reproduce the copyright notice and any other legend of
//  ownership on each copy or partial copy, of the Program.
//
//  You acknowledge and agree that the Program contains
//  copyrighted material, trade secrets and other TI proprietary
//  information and is protected by copyright laws,
//  international copyright treaties, and trade secret laws, as
//  well as other intellectual property laws.  To protect TI's
//  rights in the Program, you agree not to decompile, reverse
//  engineer, disassemble or otherwise translate any object code
//  versions of the Program to a human-readable form.  You agree
//  that in no event will you alter, remove or destroy any
//  coypright notice included in the Program.  TI reserves all
//  rights not specifically granted under this license. Except
//  as specifically provided herein, nothing in this agreement
//  shall be construed as conferring by implication, estoppel,
//  or otherwise, upon you, any license or other right under any
//  TI patents, copyrights or trade secrets.
//
//  You may not use the Program in non-TI devices.
//
//  File: emeter-foreground.c
//
//  Steve Underwood <steve-underwood@ti.com>
//  Texas Instruments Hong Kong Ltd.
//
//  $Id: emeter-foreground.c,v 1.34 2008/10/28 10:13:40 a0754793 Exp $
//
/*! \file emeter-structs.h */
//
//--------------------------------------------------------------------------
//
//  MSP430 foreground (non-interrupt) routines for e-meters
//
//  This software is appropriate for single phase and three phase e-meters
//  using a voltage sensor plus a CT or shunt resistor current sensors, or
//  a combination of a CT plus a shunt.
// 
//    Foreground process includes:
//    -Using timer tick to wait
//    -Calculating the power per channel
//    -Determine if current channel needs scaling.
//    -Determine if needs to be in low power modes.
//    -Compensate reference from temperature sensor
//
#include <stdint.h>
#include <stdlib.h>

#include <math.h>
#include "io.h"

#include <emeter-toolkit.h>
#define __MAIN_PROGRAM__

#include "emeter-structs.h"

#include "..\display\Segment_LCD.h"
#include "..\display\main.h"
#include "..\display\Menu.h"
#include "..\display\LCD.h"


const unsigned char LcdValue0[] = 
{
    0xFA,                // '0'   0xFA 
    0x60,                // '1'   0x06
    0xD6,                // '2'   0x6D
    0xF4,                // '3'   0x4F
    0x6C,                // '4'   0x6C
    0xBC,                // '5'   0xCB
    0xBE,                // '6'   0xEB
    0xE0,                // '7'   0x0E
    0xFE,                // '8'   0xEF
    0xFC,                // '9'   0xCF
    0x00,                // ȫ
};
///////////////////////////////////////////////////////////////////////////////
/////////////////////////////////Globles/////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
/*! \system configuration */    
int16_t gSysconf;
/*! \AFE functions configuration */    
int16_t gCsgconf;
/*! \power constant for pulse*/    
int16_t gPower_const;
/*! \start current */    
int16_t gStart_curr;
/* Meter status flag bits. */
uint16_t meter_status;
/* energy pulse threshold */
int32_t gCurrent_threshold;
/* check sum for calibrations data */
int16_t gChecksum1;

/* timeout for communication */
uint16_t char_timeout_1107;
/* delay for communication */
uint8_t char_timeout_interrupt;

///for test only
#define TEST_CONSISTENCY
#undef TEST_CONSISTENCY
#define RECORD_POWER
#undef RECORD_POWER
#ifdef TEST_CONSISTENCY
uint8_t buf_pt;
int32_t p_rms=0;
int32_t p_last=0;
#ifdef  RECORD_POWER
int32_t pbuf[128];
#endif
#endif

/* Current operating mode - normal, limp, power down, etc. */
//int8_t operating_mode;

/* Persistence check counters for anti-tamper measures. */
static int8_t current_reversed;
static int8_t current_unbalanced;

/* The main per-phase working parameter structure */
struct phase_parms_s chan1;

//unsigned char state; // RTCֵ״̬
//#define __infomemb__ _Pragma("location=\"INFOB\"")
//__infomemb__  


/* The main per-phase non-volatile parameter structure */
__infomem__ const struct info_mem_s nv_parms =
{
    {//seg a
        {//phase_nv_parms_s 
            {//phase_nv_parms_s chan1
                {//current_sensor_nv_parms_s live
                    0,
                    DEFAULT_I_RMS_SCALE_FACTOR_A,        //  
                    DEFAULT_P_SCALE_FACTOR_A_LOW,        //  й
                    DEFAULT_BASE_PHASE_A_CORRECTION_LOW, //  λƫе(CT,shunt)
                    DEFAULT_P_OFFSET_A_LOW,              //  йƫ(ֱSD)
                    DEFAULT_Q_OFFSET_A_LOW, 
                    //},
                    //{//current_sensor_nv_parms_s neutral
                    0,
                    DEFAULT_I_RMS_SCALE_FACTOR_NEUTRAL,  // ͬϣΪneutralΪlive
                    DEFAULT_P_SCALE_FACTOR_NEUTRAL,
                    DEFAULT_NEUTRAL_BASE_PHASE_CORRECTION,
                    DEFAULT_P_OFFSET_NEUTRAL,  
                    DEFAULT_Q_OFFSET_NEUTRAL,  
                },                
                DEFAULT_V_RMS_SCALE_FACTOR_A,         //  ѹ  
                0,
                DEFAULT_I_DC_ESTIMATE << 16,
                DEFAULT_I_DC_ESTIMATE_NEUTRAL << 16,
                DEFAULT_V_DC_ESTIMATE << 16,
            },
            {
                0,           // uint32_t total_consumed_energy;
                0,           // uint32_t total_consumed_reactive_energy;
                0,           // uint32_t total_consumed_negtive_energy;
                0,           // uint32_t total_consumed_negtive_reactive_energy; 
            },
            
            //TEMPERATURE_SUPPORT
            25,
            DEFAULT_TEMPERATURE_OFFSET,   //  y = ax+b       b
            DEFAULT_TEMPERATURE_SCALING,  //                 a 
            //CORRECTED_RTC_SUPPORT
            0,
            //rtc
            {
                0,
                0,
                0,
                0,
                0,
                0,
                0
            },
            //serial number        
            "",
            //property number        
            "",
            //factory number        
            ""
        }
    }
};

unsigned int const consumerenergy[2]@P_ENERGY_ADD =
{
    0,0,
};
///////////////////////////////////////////////////////////////////////////////
/////////////////////////////////Functions/////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
#if !defined(__IAR_SYSTEMS_ICC__)
static __inline__ long labs(long __x);
static __inline__ long labs(long __x)
{
    return (__x < 0) ? -__x : __x;
}
#endif

//Parameter discription
//correction: phase correction factor, in 1/256 of a sample time
// λУУʱʹã
void set_phase_correction(struct phase_correction_s *s, int correction, struct current_sensor_nv_parms_s const *ss)
{
    correction += 128;
    
    s->step = I_HISTORY_STEPS + (correction >> 8) - (ss->Phase_correction[0]>>8);    //V shift left(leading), so here use + because the s->step will be substract in background
    correction = 127 - ((correction & 0xFF) >> 1);
    s->fir_beta = fir_coeffs[correction][0];
    s->fir_gain = fir_coeffs[correction][1];
    /*    
    phase->current[0].quadrature_correction[0].step = I_HISTORY_STEPS + (correction >> 8) - (nv_parms.seg_a.s.chan1.current[0].Phase_correction[0]>>8);    //V shift left(leading), so here use + because the s->step will be substract in background
    correction = 127 - ((correction & 0xFF) >> 1);
    phase->current[0].quadrature_correction[0].fir_beta = fir_coeffs[correction][0];
    phase->current[0].quadrature_correction[0].fir_gain = fir_coeffs[correction][1];
    
    //neutral
    phase->current[1].quadrature_correction[0].step = I_HISTORY_STEPS + (correction >> 8) - (nv_parms.seg_a.s.chan1.current[1].Phase_correction[0]>>8);    //V shift left(leading), so here use + because the s->step will be substract in background
    correction = 127 - ((correction & 0xFF) >> 1);
    phase->current[1].quadrature_correction[0].fir_beta = fir_coeffs[correction][0];
    phase->current[1].quadrature_correction[0].fir_gain = fir_coeffs[correction][1];
    */
}

//correction: absolute phase shift  
unsigned char set_initial_sd16_phase_correction(struct phase_correction_sd16_s *s, int correction)
{
    uint8_t bump;
    
    //    bump = s->sd16_preloaded_offset - (correction & 0xFF);
    //    ȡcorrectionĵ8λȡ1
    bump = 0 - (correction & 0xFF);//this function only work for initation
    /* Always store the required correction. */
    s->step = I_HISTORY_STEPS - (correction >> 8);
    s->sd16_preloaded_offset = correction; 
    return bump;
}

static int32_t test_phase_balance(int32_t live_signal, int32_t neutral_signal, int threshold)
{
    int permitted_imbalance_fraction;
    
    /* This may be testing between two currents, or between two powers. I normal mode it
    is testing between two power readings. In limp mode it is testing between two
    current readings. */
    /* See which one is bigger, with some tolerance built in. If the signal measured
    from the neutral is more than 6.25% or 12.5% (options) different from the signal
    measured from the live there is something wrong (maybe fraudulent tampering, or
    just something faulty). In this case we use the current measured from the
    channel with the higher signal. When the channel is reasonably balanced, we use
    the signal from the live lead. If neither signal is above the threshold we use
    a more relaxed measure of imbalance (say 25% or even 50%), to allow for the lower
    accuracy of these small signals (remember, we need the test to work well under
    transient conditions, so the threshold needs to be far more relaxed than the
    basic measurement accuracy would indicate). Assessments are persistence checked
    to avoid transient conditions causing a false change of imbalance status. */
    if (live_signal <= threshold  &&  neutral_signal <= threshold)
        permitted_imbalance_fraction = RELAXED_IMBALANCE_FRACTION;
    else
        permitted_imbalance_fraction = PERMITTED_IMBALANCE_FRACTION;
    /* We have a signal strong enough for proper assessment. */
    if ((phase->status & PHASE_UNBALANCED))
    {
        /* We are looking for the restoration of balance between the
        live and neutral. */
        if ((live_signal - (live_signal >> permitted_imbalance_fraction)) < neutral_signal
            &&
                neutral_signal < (live_signal + (live_signal >> permitted_imbalance_fraction)))
        {
            /* Things might be balanced, but persistence check to be sure. */
            if (--current_unbalanced <= -PHASE_UNBALANCED_PERSISTENCE_CHECK)
            {
                /* Things look genuinely balanced. */
                phase->status &= ~(PHASE_UNBALANCED | CURRENT_FROM_NEUTRAL);
                current_unbalanced = 0;
            }
        }
        else
        {
            current_unbalanced = 0;
            /* The imbalanced might have swapped around - check. */
            /* Here we just choose the greater signal each block, as we have
            already confirmed (i.e. debounced) the imbalance condition. */
            if (neutral_signal > live_signal)
                phase->status |= CURRENT_FROM_NEUTRAL;
            else
                phase->status &= ~CURRENT_FROM_NEUTRAL;
        }
    }
    else
    {
        /* We are looking for the live and neutral becoming unbalanced. */
        if ((live_signal - (live_signal >> permitted_imbalance_fraction)) > neutral_signal
            ||
                neutral_signal > (live_signal + (live_signal >> permitted_imbalance_fraction)))
        {
            /* Things might be unbalanced, but persistence check to be sure. */
            if (++current_unbalanced >= PHASE_UNBALANCED_PERSISTENCE_CHECK)
            {
                /* Things look genuinely unbalanced. */
                current_unbalanced = 0;
                phase->status |= PHASE_UNBALANCED;
                if (neutral_signal > live_signal)
                    phase->status |= CURRENT_FROM_NEUTRAL;
                else
                    phase->status &= ~CURRENT_FROM_NEUTRAL;
            }
        }
        else
        {
            current_unbalanced = 0;
        }
    }
    /* Clear the unbalanced detection, so we don't display unbalanced. This should eliminate
    flashing of the LED if the are transient conditions causing false indications of
    imbalance. */
    if (live_signal <= threshold  &&  neutral_signal <= threshold)
        phase->status &= ~PHASE_UNBALANCED;
    if ((phase->status & CURRENT_FROM_NEUTRAL))
        return  neutral_signal;
    return  live_signal;
}

#if defined(MAINS_FREQUENCY_SUPPORT)
int16_t frequency(void)
{
    int32_t x;
    
    /* Calculate the mains frequency in 1/100Hz increments, based on the mains
    period assessment from the background activity. */
    /* Normally we get the mains frequency from the voltage. Voltage is always
    present, and is not subject to the same level of distortion as the current
    waveform with difficult loads. */
    /* We have a whole cycle period in the upper 16 bits, but we want the delay for 90 degrees, so we shift 2
    extra bits for that, because the integer part of mains_period is in units of 1/256th of a sample time. */
    
    x = (phase->mains_period >> 18);
    set_phase_correction(&phase->current[0].quadrature_correction[0], x, &nv_parms.seg_a.s.chan1.current[0]);       
    set_phase_correction(&phase->current[1].quadrature_correction[0], x, &nv_parms.seg_a.s.chan1.current[1]);
    //set_phase_correction(&phase->current[0].quadrature_correction[0], x);       
    //set_phase_correction(&phase->current[1].quadrature_correction[0], x);
    //set_phase_correction(x);
    
    x = phase->mains_period;        
    x = (int32_t) SAMPLES_PER_10_SECONDS*256L*10L/(x >> 16);
    return  x;
}
#endif

int32_t voltage(void)
{
    int32_t x;
    
    /* Calculate the RMS voltage in 10mV increments. Return -1 for overrange
    (i.e. ADC clip). */
    if ((phase->status & V_OVERRANGE))
        return -1;
    x = div_sh48(phase->V_sq_accum_logged, 26 - 2*ADC_BITS, phase->sample_count_logged);  
    x = isqrt32(x);
    x = (x >> 12)*nv_parms.seg_a.s.chan1.V_rms_scale_factor;
    x >>= 14;
    return x;
}

void current(unsigned char ch)
{
    int32_t x;
    
    /* Calculate the RMS current in 1mA increments. Return -1 for overrange
    (i.e. ADC clip). A side effect of this routine is it updates the dynamic
    phase correction settings, based on the newly calculated current. */
    /* We always have to work out the properly scaled current from both leads, in
    order to work out the FIR coeffs for the next block. */
    x = div48(phase->current[ch].I_sq_accum_logged[0], phase->sample_count_logged);
    x = isqrt32(x);     
    x = (x >> 12)*phase_nv->current[ch].I_rms_scale_factor[0];
    x >>= 14;
    //        phase->I_rms[0] = x - phase_nv->current[0].Ac_offset;
    x = x - nv_parms.seg_a.s.chan1.current[ch].Ac_offset;
    if( x > 0)
        phase->I_rms[ch] = x;
    else
        phase->I_rms[ch] = 0;        
}

int32_t active_power(void)
{
    int32_t x;
    int16_t i;
    int32_t y;
    int reversed;
    
    /* We can only do real power assessment in full operating mode. */
    /* If we have neutral monitoring for a single phase meter, we need to measure
    both power levels, and decide between them. Issues to be assessed here are
    whether one or both leads show reverse power, and whether the power levels
    are balanced. */
    
    /* If we find a negative power level we may be genuinely feeding power to the grid,
    or we may be seeing a tamper condition. This is application dependant. */
    reversed = FALSE;
    
    x = div_sh48(phase->current[0].P_accum_logged[0], 27 - 2*ADC_BITS, phase->sample_count_logged);
    //    x = div_sh48(phase->current.P_accum_logged[0], 3 + 27 - 2*ADC_BITS, phase->sample_count_logged);
    i = phase_nv->current[0].P_scale_factor[0];
    x = mul48(x, i);
    
    /////////////////////OFFSET CORRECTION//////////////////////
    x -= phase_nv->current[0].Offset_active_power;
    
    if (x < 0)
    {
        //0804      if((gCsgconf&CSGCON_EMOD_MASK)==CSGCON_EMOD2)//accumulate absolute power mode
        //        x = -x;  
        phase->status |= I_REVERSED;  
        if (x > PHASE_REVERSED_THRESHOLD_POWER)    
            reversed = TRUE;
    }
    else
    {
        phase->status &= ~I_REVERSED;   
    }
    phase->active_power[0] = x;
    
    
    ////////////////////P for neutral
    y = div_sh48(phase->current[1].P_accum_logged[0], 27 - 2*ADC_BITS, phase->sample_count_logged);
    i = phase_nv->current[1].P_scale_factor[0];
    y = mul48(y, i);
    
    /////////////////////OFFSET CORRECTION//////////////////////
    y -= neutral_nv->Offset_active_power;
    
    if (y < 0)
    {
        //0804      if((gCsgconf&CSGCON_EMOD_MASK)==CSGCON_EMOD2)//accumulate absolute power mode
        //        y = -y;
        phase->status |= I_NEUTRAL_REVERSED;
        if (y > PHASE_REVERSED_THRESHOLD_POWER)
            reversed = TRUE;
    }
    else
    {
        phase->status &= ~I_NEUTRAL_REVERSED;  
    }
    phase->active_power[1] = y;
    
    //now start persistent check for reversing
    if ((phase->status & PHASE_REVERSED))
    {
        if (!reversed)
        {
            if (--current_reversed <= -PHASE_REVERSED_PERSISTENCE_CHECK)
            {
                phase->status &= ~PHASE_REVERSED;
                current_reversed = 0;
            }
        }
        else
        {
            current_reversed = 0;
        }
    }
    else
    {
        if (reversed)
        {
            if (++current_reversed >= PHASE_REVERSED_PERSISTENCE_CHECK)
            {
                phase->status |= PHASE_REVERSED;
                current_reversed = 0;
            }
        }
        else
        {
            current_reversed = 0;
        }
    }
    
    if((gCsgconf&CSGCON_CSEL_MASK)==CSGCON_CSEL0){//unbalance checking mode, use channel which has larger current
        x = test_phase_balance(abs(x), abs(y), PHASE_UNBALANCED_THRESHOLD_POWER);
        if ((phase->status & PHASE_UNBALANCED))
        {
            /* When the phase is unbalanced we only look for reversed current in the 
            lead with the higher current. If we do not impose this restriction, coupling
            through a parasitic CT power supply transformer can cause the reverse condition
            to be raised incorrectly. If there is no parasitic supply this test is probably
            a waste of time. */
            if ((phase->status & CURRENT_FROM_NEUTRAL))
                reversed = phase->status & I_NEUTRAL_REVERSED;
            else
                reversed = phase->status & I_REVERSED;
        }
    }
    
    if(phase->status & CURRENT_FROM_NEUTRAL)
        x = y;
    
    return  x;
}

void reactive_power(void)
{
    int32_t x, y;
    int16_t i;
    
    /* We can only do real power assessment in full operating mode. */
    /* If we have neutral monitoring for a single phase meter, we need to use whichever
    channel has been selected by the anti-tamper validation scheme. */  
    ///////////neutral
    y = div_sh48(phase->current[1].P_reactive_accum_logged[0], 27 - 2*ADC_BITS, phase->sample_count_logged);
    i = phase_nv->current[1].P_scale_factor[0];
    i = Q1_15_mul(i, phase->current[1].quadrature_correction[0].fir_gain);
    y = mul48(y, i);
    y <<= 2;
    y = y - phase_nv->current[1].Offset_reactive_power;
    if (y < 0)
    {
        if((gCsgconf&CSGCON_EMOD_MASK)==CSGCON_EMOD2)//accumulate absolute power mode
            y = -y;
    }
    phase->reactive_power[1] = y;
    
    ////////////live
    x = div_sh48(phase->current[0].P_reactive_accum_logged[0], 27 - 2*ADC_BITS, phase->sample_count_logged);
    i = phase_nv->current[0].P_scale_factor[0];
    i = Q1_15_mul(i, phase->current[0].quadrature_correction[0].fir_gain);
    x = mul48(x, i);
    x <<= 2;
    x = x - phase_nv->current[0].Offset_reactive_power; 
    if (x < 0)
    {
        if((gCsgconf&CSGCON_EMOD_MASK)==CSGCON_EMOD2)//accumulate absolute power mode
            x = -x;
    }
    phase->reactive_power[0] = x;
    
}

//apparent power from live channel
int32_t apparent_power(char ch)
{
    uint32_t p;
    uint32_t x;
    int shift;
    
    /* Calculate apparent (VA) power in 0.01W increments */
    p = labs(phase->active_power[ch]);
    x = labs(phase->reactive_power[ch]);
    
    /* Justify for optimal accuracy */
    shift = 0;
    while ((p & 0xFFFF8000)  ||  (x & 0xFFFF8000))
    {
        shift++;
        p >>= 1;
        x >>= 1;
    }
    x = isqrt32(p*p + x*x);
    shift = 16-shift;
    x = x>>shift;
    //x >>= (16 - shift);
    if(x&0x80000000)
        x = 0x7fffffff;
    return  x;
}

//Power factor from live channel
int16_t power_factor(char ch)
{
    int32_t p;
    int32_t x;
    
    p = labs(phase->active_power[ch]);
    x = labs(phase->apparent_power[ch]);
    
    if (p  &&  x)
    {
        /* Justify for optimal accuracy */
        while ((p & 0x40000000) == 0  &&  (x & 0x40000000) == 0)
        {
            p <<= 1;
            x <<= 1;
        }
        x >>= 16;
        p /= x;
        p *= 10000;
        p >>= 16;
        /* Don't let a little imprecision cause strange answers */
        if (p > 10000)
            p = 10000;
    }
    else
    {
        p = 0;
    }
    /* Use a negative PF to indicate an inductive load */
    if (ch == 1)//neutral phase
    {
        if (phase->current[1].leading < 0)
            p = p;
        else
            p = -p;
    }
    else
    {
        if (phase->current[0].leading < 0)
            p = p;
        else
            p = -p;
    }
    return p;
}

///////////////get dc estimation with average machanism/////////
void dc_cal(void)
{
    int16_t x;
    
    x = phase->V_accum_logged/phase->sample_count_logged;
    phase->V_dc_estimate += x - (phase->V_dc_estimate>>DCFILTER_SHIFT);
    
    x = phase->I_accum_logged[0]/phase->sample_count_logged;
    phase->current[0].I_dc_estimate[0] += x - (phase->current[0].I_dc_estimate[0]>>DCFILTER_SHIFT);
    
    x = phase->I_accum_logged[1]/phase->sample_count_logged;
    phase->current[1].I_dc_estimate[0] += x - (phase->current[1].I_dc_estimate[0]>>DCFILTER_SHIFT);
    
}
//////////////////send interrupt to host to notify a event
void set_int_flag(unsigned char fg)
{
    phase->int_flag |= fg | CSGINTPENDING;  
}

/*
void send_interrupt()
{
P1OUT &= ~INTERRUPT_BIT;
delay(1);//2us
P1OUT |= INTERRUPT_BIT;  
}
*/

extern void DataStore(unsigned char offset, int32_t value, unsigned char bi_word);

// 20110427
// 20110503
// 20110505
// 20110506 1835
// 20110509
// 20110510
// 20110511 1029
// 20110513 1101
// 20110515 1812
// 20110526 1758 ޸˲˵ еһbug 
// unsigned char   const Tab2[] =  
//  KEY3,0,Func1, ޸Ϊ KEY3,2,Func1,
// ޸ͨѶЭ飬Ӧ

#if defined(__AQCOMPILER__)  ||  defined(__IAR_SYSTEMS_ICC__)
void main(void)
#else
int main(int argc, char *argv[])
#endif
{
    static int32_t x;
    WDTCTL = (WDTCTL  & 0xFF) | WDTPW | WDTHOLD;
    x = consumerenergy[0];   // ֹŻ
        
    //initiate meter working parameters
    data_init();
    csg_config();
    //config hardware
    system_setup();
    
    /////////////////////////////////////////////////////////////////////////////////        
    SD24PRE_LIVE = set_initial_sd16_phase_correction(&phase->current[0].in_phase_correction[0], phase_nv->current[0].Phase_correction[0]);

    for (;;)
    {
        kick_watchdog();
        if(menustatu.enter_lpm==1)
        {
            menustatu.exit_lpm = 1;
            __bis_SR_register(LPM0_bits);
        }
        
        if(menustatu.enter_lpm == 0)
        {
            if ((phase->status & NEW_LOG))   
            {
                /* The background activity has informed us that it is time to
                perform a block processing operation. */                
                phase->status &= ~NEW_LOG;        
                
                x = active_power();       //P1, P2    
                
                ///test consistency
#if defined(TEST_CONSISTENCY)          
                buf_pt++;
#ifdef  RECORD_POWER
                if(buf_pt<128)
                    pbuf[buf_pt] = x;
#endif          
                if(buf_pt>8)
                    p_rms += abs(x-p_last);
                if(buf_pt==0){
                    if(p_rms)
                        P1OUT |= 0x01;
                    p_rms = 0;
                }
                p_last = x;
#endif
                
                current(0);               //IRMS1 
                phase->V_rms = voltage(); //VRMS
                
                //    if (labs(x) < gStart_curr)  20110428
                //    if(labs(x) < gStart_curr)
                if(phase->I_rms[0] < gStart_curr)
                {
                    x = 0;                // жǷﵽ
                    phase->I_rms[0] = x;
                    phase->active_power[0] = x;
                    phase->reactive_power[0] = x;
                    phase->apparent_power[0] = x;
                    phase->power_factor[0] = 10000;
                }          
                else
                {
                    if((gCsgconf&CSGCON_QOFF)==0)//reactive power function not off
                        reactive_power();         //Q1,Q2
                    
                    if((gCsgconf&CSGCON_SOFF)==0){//apparent power function not off
                        phase->apparent_power[0] = apparent_power(0);   //S1
                        

                        /* The power factor should be calculated last */
                        phase->power_factor[0] = power_factor(0);       //PF1
                    }
                }
                phase->frequency = frequency();                 //frequency
                
                if((gCsgconf&CSGCON_IDFILTERTYPE)==CSGCON_IDFILTERTYPE)//use average filter
                    dc_cal();
                
                if( phase->int_enable & CSGENRDYFG )
                    set_int_flag(CSGENRDYFG);//notify new emeter parameters ready           
            }    
            
            if(meter_status&SPITIMEOUT){
                //char_timeout_1107 = SAMPLES_PER_10_SECONDS;     //while no communication, delay 10s to reset spi
                /*
                char_timeout_1107 = SAMPLES_PER_10_SECONDS;  // UARTʱж
                */
                char_timeout_1107 = 409;
                meter_status &= ~SPITIMEOUT;
//                init_serial_port();
            }
            
            
            /* ͨжSDֵȷֵGAIN ñ1 */
            if(menustatu.key_value!=0)               // м 
            {
                MenuMonitor(menustatu.key_value);    // ˵
                menustatu.key_value = 0;             // ϣ尴
            }
        }
        /* RTC */
        if(menustatu.rtc_bump_fg==1)             
        {
            menustatu.rtc_bump_fg = 0;
            bump_rtc();
            UpDataDisInfo(UpdataFg);
        }
        //__bis_SR_register(LPM0_bits);
    }// end of for(;;)
}